<?php

/*
 * This file is part of the Flash Sale plugin
 *
 * Copyright(c) ECCUBE VN LAB. All Rights Reserved.
 *
 * https://www.facebook.com/groups/eccube.vn
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Plugin\ApgSimpleCalendar;

use Doctrine\ORM\EntityManagerInterface;
use Eccube\Common\Constant;
use Eccube\Entity\Block;
use Eccube\Entity\Master\DeviceType;
use Eccube\Plugin\AbstractPluginManager;
use Eccube\Repository\BlockRepository;
use Eccube\Repository\Master\DeviceTypeRepository;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Filesystem\Filesystem;

class PluginManager extends AbstractPluginManager
{

    const BLOCK_NAME = [

        '営業日カレンダー(当月のみ)' => 'apg_sc_single_calendar',
        '営業日カレンダー(当月、翌月)' => 'apg_sc_two_month_calendar',
    ];

    private $blockBasePath;


    /**
     * PluginManager constructor.
     */
    public function __construct()
    {
        $this->blockBasePath = __DIR__ . '/Resource/template/block/';
    }

    /**
     * @param array $meta
     * @param ContainerInterface $container
     *
     * @throws \Exception
     */
    public function uninstall(array $meta, ContainerInterface $container)
    {
        foreach (self::BLOCK_NAME as $blockName => $fileName) {
            $this->removeDataBlock($container, $fileName);
        }
        /** @var EntityManagerInterface $entityManager */
        $entityManager = $container->get('doctrine')->getManager();
        $this->migration($entityManager->getConnection(), $meta['code'], '0');
    }

    /**
     * @param null|array $meta
     * @param ContainerInterface $container
     *
     * @throws \Exception
     */
    public function enable(array $meta = null, ContainerInterface $container)
    {
        foreach (self::BLOCK_NAME as $blockName => $fileName) {
            $this->copyBlock($container, $fileName);
            $Block = $container->get(BlockRepository::class)->findOneBy(['file_name' => $fileName]);
            if (is_null($Block)) {
                $this->createDataBlock($container, $blockName, $fileName);
            }
        }
        /** @var EntityManagerInterface $entityManager */
        $entityManager = $container->get('doctrine')->getManager();
        $this->migration($entityManager->getConnection(), $meta['code']);
    }

    /**
     * @param array|null $meta
     * @param ContainerInterface $container
     *
     * @throws \Exception
     */
    public function disable(array $meta = null, ContainerInterface $container)
    {
        foreach (self::BLOCK_NAME as $blockName => $fileName) {
            $this->removeDataBlock($container, $fileName);
            $this->removeBlock($container, $fileName);
        }
    }

    /**
     * @param array|null $meta
     * @param ContainerInterface $container
     */
    public function update(array $meta = null, ContainerInterface $container)
    {
        foreach (self::BLOCK_NAME as $blockName => $fileName) {
            $this->copyBlock($container, $fileName);
        }
        /** @var EntityManagerInterface $entityManager */
        $entityManager = $container->get('doctrine')->getManager();
        $this->migration($entityManager->getConnection(), $meta['code']);
    }

    /**
     * @param ContainerInterface $container
     *
     * @throws \Exception
     */
    private function createDataBlock(ContainerInterface $container, $blockName, $fileName)
    {
        $em = $container->get('doctrine.orm.entity_manager');
        $DeviceType = $container->get(DeviceTypeRepository::class)->find(DeviceType::DEVICE_TYPE_PC);
        try {
            /** @var Block $Block */
            $Block = $container->get(BlockRepository::class)->newBlock($DeviceType);
            $Block->setName($blockName)
                ->setFileName($fileName)
                ->setUseController(Constant::DISABLED);
            $em->persist($Block);
            $em->flush($Block);
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * @param ContainerInterface $container
     *
     * @throws \Exception
     */
    private function removeDataBlock(ContainerInterface $container, $fileName)
    {
        /** @var \Eccube\Entity\Block $Block */
        $Block = $container->get(BlockRepository::class)->findOneBy(['file_name' => $fileName]);
        if (!$Block) {
            return;
        }
        $em = $container->get('doctrine.orm.entity_manager');
        try {
            $blockPositions = $Block->getBlockPositions();
            /** @var \Eccube\Entity\BlockPosition $BlockPosition */
            foreach ($blockPositions as $BlockPosition) {
                $Block->removeBlockPosition($BlockPosition);
                $em->remove($BlockPosition);
            }
            $em->remove($Block);
            $em->flush();
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * @param ContainerInterface $container
     * @param $fileName
     */
    private function copyBlock(ContainerInterface $container, $fileName)
    {
        $templateDir = $container->getParameter('eccube_theme_front_dir');
        $file = new Filesystem();
        $filePath = $this->blockBasePath . $fileName . '.twig';
        $file->copy($filePath, $templateDir . '/Block/' . $fileName . '.twig');
    }

    /**
     * Remove block template.
     *
     * @param ContainerInterface $container
     */
    private function removeBlock(ContainerInterface $container, $fileName)
    {
        $templateDir = $container->getParameter('eccube_theme_front_dir');
        $file = new Filesystem();
        $file->remove($templateDir . '/Block/' . $fileName . '.twig');
    }

}
